/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:    All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.    The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true, node: true, expr: true*/
/*global describe: true, expect: true, it: true, afterEach: true, beforeEach: true */
"use strict";

var PreviewRequestHandler = require("../lib/preview-request-handler"),
    fs = require("fs"),
    sinon = require("sinon"),
    Buffer = require("buffer").Buffer,
    Stream = require("stream");

describe("PreviewRequestHandler", function() {
    var sandbox = sinon.sandbox.create(),
        previewPath = "/tmp/foo.jpg",
        previewBuffer = new Buffer([0,1,2,3,4,5,6,7]),
        previewId = 10,
        uri = "/preview/10.jpg",
        bitmapContentType = "image/jpeg",
        svgContentType = "image/svg+xml",
        response,
        getPreviewPathFunction = sinon.stub().returns({ tempAssetPath: previewPath }),
        getPreviewBufferFunction = sinon.stub().returns({ buffer: previewPath }),
        readStream;

    beforeEach(function () {
        response = {
            write: sinon.stub(),
            writeHead: sinon.stub(),
            end: sinon.stub()
        };
        readStream = {
            pipe: sinon.stub(),
            _read: sinon.stub()
        };
        sandbox.stub(fs, "createReadStream").returns(readStream);
        sandbox.stub(Stream, "Readable").returns(readStream);
    });

    afterEach(function () {
        sandbox.restore();
    });

    describe("successful temp file requests", function () {
        function expectResponseForValidBitmapPreview() {
            expect(response.writeHead).to.have.been.calledOnce;
            expect(response.writeHead).to.have.been.calledWith(200, { "Content-Type": bitmapContentType });
            expect(fs.createReadStream).to.have.been.calledWith(previewPath);
            expect(readStream.pipe).to.have.been.calledOnce;
            expect(readStream.pipe).to.have.been.calledWith(response);
            expect(response.write).to.not.have.been.called;
            expect(response.end).to.not.have.been.calledOnce;
        }

        function expectResponseForValidSVGPreview() {
            expect(response.writeHead).to.have.been.calledOnce;
            expect(response.writeHead).to.have.been.calledWith(200, {
                "Content-Type": svgContentType,
                "Vary": "Accept-Encoding"
            });
            expect(fs.createReadStream).to.have.been.calledWith(previewPath, { encoding: "utf8" });
            expect(readStream.pipe).to.have.been.calledOnce;
            expect(readStream.pipe).to.have.been.calledWith(response);
            expect(response.write).to.not.have.been.called;
            expect(response.end).to.not.have.been.calledOnce;
        }

        it("responds successfully for a valid bitmap preview", function () {
            PreviewRequestHandler.handlePreviewRequest(uri, response, bitmapContentType, getPreviewPathFunction);
            expectResponseForValidBitmapPreview();
        });

        it("responds successfully for a valid SVG preview", function () {
            PreviewRequestHandler.handlePreviewRequest(uri, response, svgContentType, getPreviewPathFunction);
            expectResponseForValidSVGPreview();
        });

        it("responds successfully even if the preview id has no dot plus extension", function () {
            PreviewRequestHandler.handlePreviewRequest("/preview/10", response, bitmapContentType, getPreviewPathFunction);
            expectResponseForValidBitmapPreview();
        });

        it("responds successfully even if the preview id has no extension", function () {
            PreviewRequestHandler.handlePreviewRequest("/preview/10.", response, bitmapContentType, getPreviewPathFunction);
            expectResponseForValidBitmapPreview();
        });

        it("responds successfully even if the preview id has an invalid extension", function () {
            PreviewRequestHandler.handlePreviewRequest("/preview/10.xyz", response, bitmapContentType, getPreviewPathFunction);
            expectResponseForValidBitmapPreview();
        });
    });

    describe("successful buffer requests", function () {
        function expectResponseForValidBitmapPreview() {
            expect(response.writeHead).to.have.been.calledOnce;
            expect(response.writeHead).to.have.been.calledWith(200, { "Content-Type": bitmapContentType });
            expect(Stream.Readable).to.have.been.calledOnce;
            expect(readStream.pipe).to.have.been.calledOnce;
            expect(readStream.pipe).to.have.been.calledWith(response);
            expect(response.write).to.not.have.been.called;
            expect(response.end).to.not.have.been.calledOnce;
        }

        function expectResponseForValidSVGPreview() {
            expect(response.writeHead).to.have.been.calledOnce;
            expect(response.writeHead).to.have.been.calledWith(200, {
                "Content-Type": svgContentType,
                "Vary": "Accept-Encoding"
            });
            expect(Stream.Readable).to.have.been.calledWith({ encoding: "utf8" });
            expect(readStream.pipe).to.have.been.calledOnce;
            expect(readStream.pipe).to.have.been.calledWith(response);
            expect(response.write).to.not.have.been.called;
            expect(response.end).to.not.have.been.calledOnce;
        }

        it("responds successfully for a valid bitmap preview", function () {
            PreviewRequestHandler.handlePreviewRequest(uri, response, bitmapContentType, getPreviewBufferFunction);
            expectResponseForValidBitmapPreview();
        });

        it("responds successfully for a valid SVG preview", function () {
            PreviewRequestHandler.handlePreviewRequest(uri, response, svgContentType, getPreviewBufferFunction);
            expectResponseForValidSVGPreview();
        });

        it("responds successfully even if the preview id has no dot plus extension", function () {
            PreviewRequestHandler.handlePreviewRequest("/preview/10", response, bitmapContentType, getPreviewBufferFunction);
            expectResponseForValidBitmapPreview();
        });

        it("responds successfully even if the preview id has no extension", function () {
            PreviewRequestHandler.handlePreviewRequest("/preview/10.", response, bitmapContentType, getPreviewBufferFunction);
            expectResponseForValidBitmapPreview();
        });

        it("responds successfully even if the preview id has an invalid extension", function () {
            PreviewRequestHandler.handlePreviewRequest("/preview/10.xyz", response, bitmapContentType, getPreviewBufferFunction);
            expectResponseForValidBitmapPreview();
        });
    });

    describe("unsuccessful requests", function () {
        function expectResponseForMalformedRequest() {
            expect(response.writeHead).to.have.been.calledOnce;
            expect(response.writeHead).to.have.been.calledWith(400, { "Content-Type": "text/plain" });
            expect(response.write).to.have.been.calledOnce;
            expect(response.write).to.have.been.calledWith("malformed route\n");
            expect(response.end).to.have.been.calledOnce;
        }

        function expectResponseForMissingPreview() {
            expect(response.writeHead).to.have.been.calledOnce;
            expect(response.writeHead).to.have.been.calledWith(404, { "Content-Type": "text/plain" });
            expect(response.write).to.have.been.calledOnce;
            expect(response.write).to.have.been.calledWith("preview not found\n");
            expect(response.end).to.have.been.calledOnce;
        }

        it("responds with an error if the resource name in the uri is malformed", function () {
            var malformedURI = "/bad-resource/uri.jpg";
            PreviewRequestHandler.handlePreviewRequest(malformedURI, response, bitmapContentType, getPreviewPathFunction);
            expectResponseForMalformedRequest();
        });

        it("responds with an error if the preview id in the uri is malformed", function () {
            var malformedURI = "/preview/not-a-number.jpg";
            PreviewRequestHandler.handlePreviewRequest(malformedURI, response, bitmapContentType, getPreviewPathFunction);
            expectResponseForMalformedRequest();
        });

        it("responds with an error if there is no preview for the requested id", function () {
            var getPreviewFunction = sinon.stub().returns(null);
            PreviewRequestHandler.handlePreviewRequest(uri, response, bitmapContentType, getPreviewFunction);
            expectResponseForMissingPreview();
        });
    });
});
